// binary file viewer Puting
//
// Copyright © 2023 Yamada Yohei <yamadayohei@gmail.com>
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the “Software”), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
// sell copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED “AS IS”, WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
// IN THE SOFTWARE.

#ifndef UNICODE
#define UNICODE
#endif
#define WINVER 0xA00
#include <windows.h>
#include <wchar.h>
#include <stdio.h>
#define B_MAIN TEXT("b_main")
#define DEFAULT_FONT_HEIGHT 20

enum {
   PIECE_ORIG,
   PIECE_APPEND,
};
struct piece {
   size_t offs;
   size_t size;
   int which;
};

enum {
   INPUT_CHOICE_NORMAL,
   INPUT_CHOICE_OCT,
   INPUT_CHOICE_HEX,
};
struct buffer {
   struct piece *table;
   char *orig_mem;
   char *append_mem;
   size_t memsize;
   size_t tablesize;
   size_t append_size;

   int n, m;
   int input_choice;
   int pos;
   int cursor;
   PTSTR filename;
   HWND h_wnd;
   int win_width;
   int win_height;
}        *g_bufs;
int       g_n_bufs;
int       g_darkmode;
HBRUSH    g_brush_dark_bg;
HINSTANCE g_hinst;
enum {
   CODE_KOI8,
   CODE_LATIN1,
   CODE_ASCII,
   CODE_END,
}         g_charcode;
int       g_font_height = DEFAULT_FONT_HEIGHT;
int       g_dpi_scale = 10;

#define RGB12(x) RGB \
   ( ((x >> 8) & 0xf) * 0x11, \
     ((x >> 4) & 0xf) * 0x11, \
     ((x >> 0) & 0xf) * 0x11  )

const COLORREF green   = RGB12(0x3c6);
const COLORREF grey    = RGB12(0x999);
const COLORREF red     = RGB12(0xf03);
const COLORREF yellow  = RGB12(0xfc3);
const COLORREF dark_bg = RGB12(0x222);

int is_darkmode(void);

// char code

wchar_t
byte_to_koi8(unsigned int n)
{
   n -= 0x80;
   const wchar_t tbl[] = {
      0x2500, 0x2502, 0x250c, 0x2510, 0x2514, 0x2518, 0x251c, 0x2524,
      0x252c, 0x2534, 0x253c, 0x2580, 0x2584, 0x2588, 0x258c, 0x2590,
      0x2591, 0x2592, 0x2593, 0x2320, 0x25a0, 0x2219, 0x221a, 0x2248,
      0x2264, 0x2265, '.',    0x2321, 0x00b0, 0x00b2, 0x00b7, 0x00f7,
      0x2550, 0x2551, 0x2552, 0x0451, 0x2553, 0x2554, 0x2555, 0x2556,
      0x2557, 0x2558, 0x2559, 0x255a, 0x255b, 0x255c, 0x255d, 0x255e,
      0x255f, 0x2560, 0x2561, 0x0401, 0x2562, 0x2563, 0x2564, 0x2565,
      0x2566, 0x2567, 0x2568, 0x2569, 0x256a, 0x256b, 0x256c, 0x00a9,

      0x44e, 0x430, 0x431, 0x446, 0x434, 0x435, 0x444, 0x433,
      0x445, 0x438, 0x439, 0x43a, 0x43b, 0x43c, 0x43d, 0x43e,
      0x43f, 0x44f, 0x440, 0x441, 0x442, 0x443, 0x436, 0x432,
      0x44c, 0x44b, 0x437, 0x448, 0x44d, 0x449, 0x447, 0x44a,

      0x42e, 0x410, 0x411, 0x426, 0x414, 0x415, 0x424, 0x413,
      0x425, 0x418, 0x419, 0x41a, 0x41b, 0x41c, 0x41d, 0x41e,
      0x41f, 0x42f, 0x420, 0x421, 0x422, 0x423, 0x416, 0x412,
      0x42c, 0x42b, 0x417, 0x428, 0x42d, 0x429, 0x427, 0x42a,
   };
   return tbl[n];
}

wchar_t
byte_to_char(unsigned int n)
{
   switch (g_charcode) {
   case CODE_ASCII:
      if (n < ' ' || n >= 0x7f) n = '.';
      if (n == ' ') n = 0xa0;
      break;
   case CODE_LATIN1:
      if (n < ' ') n = '.';
      if (n >= 0x7f && n < 0xa0) n = '.';
      if (n == ' ') n = 0xa0;
      break;
   case CODE_KOI8:
      if (n < ' ')   n = '.';
      if (n == 0x7f) n = '.';
      if (n >= 0x80) return byte_to_koi8(n);
      break;
   }
   return (wchar_t)n;
}

// buffer access

unsigned char
at(struct buffer *buf, size_t offset)
{
   if (offset >= buf->memsize) return 0;
   for (int i = 0; i < buf->tablesize; i++) {
      struct piece *p = &buf->table[i];
      if (offset < p->size) {
         unsigned char *mem = p->which == PIECE_ORIG ?
            buf->orig_mem : buf->append_mem;
         return mem[p->offs + offset]; }
      else
         offset -= p->size; }
   return 0;
}

void
copy_from_buf(struct buffer *buf, size_t offset, size_t n, unsigned char *to)
{
   for (int i = 0; i < n; i++)
      to[i] = at(buf, offset + i);
}

int
current_piece(struct buffer *buf)
{
   size_t offset = buf->cursor;
   for (int i = 0; i < buf->tablesize; i++) {
      struct piece *p = &buf->table[i];
      if (offset < p->size)
         return i;
      offset -= p->size; }
   return -1;
}

int
current_pos_in_piece(struct buffer *buf)
{
   size_t offset = buf->cursor;
   for (int i = 0; i < buf->tablesize; i++) {
      struct piece *p = &buf->table[i];
      if (offset < p->size)
         return offset;
      offset -= p->size; }
   return -1;
}

void
insert_empty_piece(struct buffer *buf, int index)
{
   buf->tablesize++;

   buf->table = (struct piece *)realloc(buf->table, buf->tablesize * sizeof(*buf->table));
   if (!buf->table) abort();
   for (int i = buf->tablesize - 1; i > index; i--)
      buf->table[i] = buf->table[i - 1];
   buf->table[index].which = PIECE_ORIG;
   buf->table[index].offs = 0;
   buf->table[index].size = 0;
}

void
delete_empty_piece(struct buffer *buf, int index)
{
   if (index < 0 || index >= buf->tablesize) abort();
   if (buf->table[index].size > 0) return;
   for (int i = index; i < buf->tablesize - 1; i++)
      buf->table[i] = buf->table[i + 1];
   buf->tablesize--;
   buf->table = (struct piece *)realloc(buf->table, buf->tablesize * sizeof(*buf->table));
   if (!buf->table) abort();
}

void
buf_delete_char(struct buffer *buf)
{
   if (buf->cursor >= buf->memsize) return;
   buf->memsize--;

   int cur = current_piece(buf);
   struct piece *p = &buf->table[cur];
   size_t offs = current_pos_in_piece(buf);
   if (!offs) { // piece head
      p->offs++;
      p->size--;
      if (!p->size) // last one char of this piece
         delete_empty_piece(buf, cur); }
   else if (buf->table[cur].size == offs + 1) // piece tail
      p->size--;
   else {
      insert_empty_piece(buf, cur + 1);
      buf->table[cur + 1].which = buf->table[cur].which;
      buf->table[cur + 1].offs = buf->table[cur].offs + offs + 1;
      buf->table[cur + 1].size = buf->table[cur].size - offs - 1;
      buf->table[cur].size = offs; }
}

void
buf_delete_chars(struct buffer *buf, int n)
{
   if (n < 1) n = 1;
   for (int i = 0; i < n; i++)
      buf_delete_char(buf);
}

void
buf_insert_char(struct buffer *buf, unsigned char c)
{
   if (buf->cursor > buf->memsize) return;
   buf->memsize++;

   int cur = current_piece(buf);
   struct piece *p = &buf->table[cur];
   size_t offs = current_pos_in_piece(buf);
   if (offs) {
      insert_empty_piece(buf, cur + 1);
      buf->table[cur + 1].which = buf->table[cur].which;
      buf->table[cur + 1].offs = buf->table[cur].offs + offs;
      buf->table[cur + 1].size = buf->table[cur].size - offs;
      buf->table[cur].size = offs;
      cur++; }

   buf->append_mem = realloc(buf->append_mem, buf->append_size + 1);
   buf->append_mem[buf->append_size++] = c;
   insert_empty_piece(buf, cur);
   buf->table[cur].which = PIECE_APPEND;
   buf->table[cur].offs = buf->append_size - 1;
   buf->table[cur].size = 1;

   buf->cursor++;
}

void
con_show_piece_table(struct buffer *buf)
{
   fprintf(stderr, "\nbuf %x n=%d\n", buf, buf->tablesize);
   for (int i = 0; i < buf->tablesize; i++) {
      struct piece *p = &buf->table[i];
      fprintf(stderr, "%7d %7d %s\n", p->offs, p->size,
            p->which == PIECE_ORIG ? "orig" : "append"); }
}

// dump

void
dump(HDC h, struct buffer *buf)
{
   HFONT h_font = CreateFont(
         /*height=*/g_font_height, /*width=*/0, /*escapement=*/0, /*orientation=*/0,
         /*weight=*/FW_NORMAL,
         /*itaric=*/FALSE, /*underline=*/FALSE, /*strike=*/FALSE,
         /*charset=*/ANSI_CHARSET,
         OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY,
         FIXED_PITCH | FF_DONTCARE, TEXT("Consolas"));
   if (h_font) SelectObject(h, h_font);

   TCHAR s[10];
   const int line_height = 24 * g_dpi_scale / 10.0;
   const int n_lines = buf->win_height / line_height;

   const COLORREF old_fg = SetTextColor(h, RGB(0x11, 0x11, 0x33));
   const COLORREF old_bg = SetBkColor(h, RGB(0xff, 0xff, 0xee));

   if (g_darkmode) {
      SetTextColor(h, RGB(0xcc, 0xcc, 0xcc));
      SetBkColor(h, dark_bg); }

   // main dump in hex 16 byte per line
   for (int i = 0; i < n_lines; i++) {
      int x = 0, y = line_height * i;

      // address
      swprintf(s, 10, TEXT("%08x"), (i + buf->pos) * 16);
      size_t l = lstrlen(s);
      TextOut(h, x + 16, y + 16, s, l);

      // in hex
      for (int j = 0; j < 16; j++) {
         int x = (32 * j + 120) * g_dpi_scale / 10.0;
x += j / 8 * 8;
x += j / 4 * 8;
x += j / 2 * 0;
         int ij = (i + buf->pos) * 16 + j;
         unsigned int n = 0;
         if (ij < buf->memsize) {
            n = 0xff & (unsigned int)at(buf, ij);
            swprintf(s, 3, TEXT("%02x"), n); }
         else
            swprintf(s, 3, TEXT("--"));
         size_t l = lstrlen(s);
         TextOut(h, x, y + 16, s, l); }

      // as char
      for (int j = 0; j < 16; j++) {
         const int x = (16 * j + 700) * g_dpi_scale / 10.0;
         const int ij = (i + buf->pos) * 16 + j;
         unsigned int n = 0;
         if (ij < buf->memsize)
            n = 0xff & (unsigned int)at(buf, ij);
         const wchar_t c = byte_to_char(n);
         TextOut(h, x, y + 16, &c, 1); } }

   // dump value at cursor as several types
   if (buf->cursor >= 0 && buf->cursor < buf->memsize) {
      int i = 0;
      const int x = 1000 * g_dpi_scale / 10.0 + 200;
      TCHAR s[64];
      unsigned char p[4];
      copy_from_buf(buf, buf->cursor, 4, p);

      unsigned int n = buf->cursor;
      swprintf(s, 64, TEXT("cursor: %08x"), n);
      size_t l = lstrlen(s);
      TextOut(h, x, i++ * 24 + 16, s, l);

      n = 0xff & *p;
      swprintf(s, 64, TEXT("u8:    %02x (%u)   "), n, n);
      l = lstrlen(s);
      TextOut(h, x, i++ * 24 + 16, s, l);

      n |= (0xff & p[1]) << 8;
      swprintf(s, 64, TEXT("u16le: %04x (%u)     "), n, n);
      l = lstrlen(s);
      TextOut(h, x, i++ * 24 + 16, s, l);

      n |= (0xff & p[2]) << 16;
      n |= (0xff & p[3]) << 24;
      swprintf(s, 64, TEXT("u32le: %08x (%u)          "), n, n);
      l = lstrlen(s);
      TextOut(h, x, i++ * 24 + 16, s, l);

      n = (0xff & p[0]) << 8 | 0xff &p[1];
      swprintf(s, 64, TEXT("u16be: %04x (%u)     "), n, n);
      l = lstrlen(s);
      TextOut(h, x, i++ * 24 + 16, s, l);

      n = n << 16 | (0xff & p[2]) << 8 | 0xff &p[3];
      swprintf(s, 64, TEXT("u32be: %08x (%u)          "), n, n);
      l = lstrlen(s);
      TextOut(h, x, i++ * 24 + 16, s, l);

      swprintf(s, 64, buf->m && buf->input_choice == INPUT_CHOICE_OCT ?
         L"m: %08x (0%o)" :
         L"m: %08x (%d)            ", buf->m, buf->m);
      l = lstrlen(s);
      TextOut(h, x, i++ * 24 + 16, s, l);

      swprintf(s, 64, buf->n && buf->input_choice == INPUT_CHOICE_OCT ?
         L"n: %08x (0%o)" :
         L"n: %08x (%d)            ", buf->n, buf->n);
      l = lstrlen(s);
      TextOut(h, x, i++ * 24 + 16, s, l);

      n = g_dpi_scale * 14.4;
      swprintf(s, 64, TEXT("dpi: %u  "), n);
      l = lstrlen(s);
      TextOut(h, x, i++ * 24 + 16, s, l);

      n = (unsigned int)GetCurrentProcessId();
      swprintf(s, 64, TEXT("pid: %u"), n);
      l = lstrlen(s);
      TextOut(h, x, i++ * 24 + 16, s, l);
   }

   SetTextColor(h, old_fg);
   SetBkColor(h, old_bg);

   // show cursor
   const int cursor = buf->cursor - buf->pos * 16;
   if (cursor >= 0 && cursor < n_lines * 16 && buf->cursor <= buf->memsize) {
      int x = (cursor % 16 * 32 + 120) * g_dpi_scale / 10.0;
      const int j = cursor % 16;
x += j / 8 * 8;
x += j / 4 * 8;
x += j / 2 * 0;
      const int y = cursor / 16 * line_height;
      const unsigned int n = 0xff & (unsigned int)at(buf, buf->cursor);
      swprintf(s, 3, TEXT("%02x"), n);
      if (buf->cursor == buf->memsize) swprintf(s, 3, L"--");
      size_t l = lstrlen(s);
      const COLORREF old_fg = SetTextColor(h, RGB(0xff, 0xff, 0xff));
      const COLORREF old_bg = SetBkColor(h, RGB(0x33, 0x33, 0x66));

      if (g_darkmode) {
         SetTextColor(h, RGB(0x11, 0x11, 0x33));
         SetBkColor(h, RGB(0xee, 0xee, 0xff)); }

      TextOut(h, x, y + 16, s, l);
      SetTextColor(h, old_fg);
      SetBkColor(h, old_bg);
   }

   // end of drawing chars
   SelectObject(h, GetStockObject(SYSTEM_FONT));
   if (h_font) DeleteObject(h_font);

   // graphic dump
   HBRUSH brush_red    = CreateSolidBrush(red);
   HBRUSH brush_grey   = CreateSolidBrush(grey);
   HBRUSH brush_green  = CreateSolidBrush(green);
   HBRUSH brush_yellow = CreateSolidBrush(yellow);
   HBRUSH brush_white  = GetStockObject(WHITE_BRUSH);
   HGDIOBJ old_pen   = SelectObject(h, GetStockObject(WHITE_PEN));
   HGDIOBJ old_brush = SelectObject(h, GetStockObject(WHITE_BRUSH));
   HPEN pen_bg = CreatePen(PS_SOLID, 0, dark_bg);

   if (g_darkmode) {
      SelectObject(h, pen_bg);
      SelectObject(h, GetStockObject(BLACK_BRUSH)); }

   // drawing bars 1
   const int n_grph_lines = buf->win_height / 4;
   const int n_grph_top = buf->pos < n_grph_lines / 2 ? n_grph_lines / 2 : buf->pos;
   const int x = 1000 * g_dpi_scale / 10.0 - 8;
   const int y = (buf->pos - n_grph_top + n_grph_lines / 2) * 4 + 16;
   SelectObject(h, g_darkmode ? g_brush_dark_bg : brush_white);
   Rectangle(h, x, 0, x + 4, buf->win_height);
   SelectObject(h, brush_yellow);
   Rectangle(h, x, y, x + 4, y + n_lines * 4);

   // drawing bars 2
   const int x2 = x + 8 * 16 + 32;
   SelectObject(h, brush_grey);
   Rectangle(h, x2, 16 - 3, x2 + 8, buf->win_height + 16 + 3);
   if (buf->win_height > 64) {
      const int Y = 16 * buf->pos;
      const int H = 16 * n_lines;
      const int top = 16  + (buf->win_height - 0) * (float)Y / buf->memsize;
      const int bot = top + (buf->win_height - 0) * (float)H / buf->memsize;
      SelectObject(h, g_darkmode ? g_brush_dark_bg : brush_white);
      Rectangle(h, x2 + 2, top, x2 + 6, bot); }

   // drawing graphic dump
   for (int i = 0; i < n_grph_lines; i++) {
      const int y = 4 * i + 16;
      const int i2 = i - n_grph_lines / 2;

      for (int j = 0; j < 16; j++) {
         const int x = 8 * j + 1000 * g_dpi_scale / 10.0;
         const int ij = (i2 + n_grph_top) * 16 + j;
         const int c = ij >= 0 && ij < buf->memsize ? 0xff & at(buf, ij) : 0;

         switch (c) {
         case 0:
            SelectObject(h, g_darkmode ? g_brush_dark_bg : brush_white);
            break;
         case 1 ... 0x20:
         case 0x7f:
            SelectObject(h, brush_red);
            break;
         case 0x21 ... 0x7e:
            SelectObject(h, brush_grey);
            break;
         case 0x80 ... 0xa0:
         case 0xff:
            SelectObject(h, brush_green);
            break;
         case 0xa1 ... 0xfe:
            SelectObject(h, brush_yellow);
            break;
         }
         Rectangle(h, x, y, x + 8, y + 4); } }

   SelectObject(h, old_pen);
   SelectObject(h, old_brush);
   DeleteObject(brush_red);
   DeleteObject(brush_green);
   DeleteObject(brush_yellow);
   DeleteObject(pen_bg);
}

// buffer management functions

struct buffer *
find_buf_from_hwnd(HWND h)
{
   for (int i = 0; i < g_n_bufs; i++)
      if (g_bufs[i].h_wnd == h)
         return &g_bufs[i];

   return g_bufs; // to avoid NULL access
}

struct buffer *
create_buffer(void)
{
   g_bufs = realloc(g_bufs, (g_n_bufs + 1) * sizeof(*g_bufs));
   if (!g_bufs) return NULL;

   struct buffer *buf = &g_bufs[g_n_bufs++];
   memset(buf, 0, sizeof(*buf));
   return buf;
}

void
delete_buffer(struct buffer *buf)
{
   if (buf->orig_mem) UnmapViewOfFile(buf->orig_mem);
   *buf = g_bufs[--g_n_bufs];
}

int mapping(struct buffer *buf);
HWND create_bmain_window(HINSTANCE h_inst, PCTSTR filename);

#define dnd_num_of_files(h)      DragQueryFile(h, ~(UINT)0, NULL, 0)
#define dnd_num_of_chars(h, i)   DragQueryFile(h, i,        NULL, 0)
#define dnd_filename(h, i, f, s) DragQueryFile(h, i, f, s)

void
open_files(HDROP h_drop)
{
   const int n_files = dnd_num_of_files(h_drop);

   for (int i = 0; i < n_files; i++) {
      struct buffer *buf = create_buffer();
      const int n = dnd_num_of_chars(h_drop, i) + 1;
      wchar_t *filename = malloc(n * sizeof(wchar_t));
      dnd_filename(h_drop, i, filename, n);
      buf->filename = filename;
      if (mapping(buf)) {
         MessageBox(NULL, filename, TEXT("Puting DnD: failed to open file"), MB_OK);
         delete_buffer(buf);
         free(filename);
         continue; }

      HWND h_wnd = create_bmain_window(g_hinst, buf->filename);
      if (!h_wnd) {
         MessageBox(NULL, filename, TEXT("Puting DnD: failed to open window"), MB_OK);
         delete_buffer(buf);
         free(filename); }
      else {
         buf->h_wnd = h_wnd;
         ShowWindow(h_wnd, SW_SHOW); } }
}

#define CTRL(x) (x & 0x1f)

LRESULT CALLBACK
wproc(HWND w, UINT m, WPARAM wp, LPARAM lp)
{
   struct buffer *buf = find_buf_from_hwnd(w);
   const int line_height = 24 * g_dpi_scale / 10.0;
   const int n_lines = buf->win_height / line_height;
   const int n = buf->n ?: 1;

   switch (m) {
   case WM_CHAR:
      InvalidateRect(w, NULL, FALSE);

      // oct
      if (buf->input_choice == INPUT_CHOICE_OCT)
         switch ((TCHAR)wp) {
         case L'0'...L'7': buf->n = buf->n * 010 + (TCHAR)wp - L'0'; return 0;
         case L'8'...L'9': return 0; }

      // oct
      if (buf->input_choice == INPUT_CHOICE_NORMAL && !buf->n)
         switch ((TCHAR)wp) {
         case L'0': buf->input_choice = INPUT_CHOICE_OCT; return 0; }

      // hex
      if (buf->input_choice == INPUT_CHOICE_HEX)
         switch ((TCHAR)wp) {
         case L'0'...L'9': buf->n = buf->n * 0x10 + (TCHAR)wp - L'0'; return 0;
         case L'a'...L'f': buf->n = buf->n * 0x10 + (TCHAR)wp - L'a' + 0xa; return 0;
         case L'A'...L'F': buf->n = buf->n * 0x10 + (TCHAR)wp - L'A' + 0xa; return 0; }

      switch ((TCHAR)wp) {
      // num
      case L'0'...L'9': buf->n = buf->n * 10 + (TCHAR)wp - L'0'; return 0;
      case L',': buf->m = buf->n, buf->n = 0, buf->input_choice = INPUT_CHOICE_NORMAL; return 0;
      case L'$': buf->input_choice = buf->input_choice == INPUT_CHOICE_HEX ? INPUT_CHOICE_NORMAL : INPUT_CHOICE_HEX; return 0;

      // system
      case L'q': // FALLTHRU
      case CTRL('w'): DestroyWindow(w); break;
      case L'Q': PostQuitMessage(0); break;
      case CTRL('l'): InvalidateRect(w, NULL, TRUE); break;
      case CTRL('y'): g_charcode++; g_charcode %= CODE_END; break;

      // edit
      case L'x': buf_delete_chars(buf, buf->n); break;
      case L'X': con_show_piece_table(buf); break; // DEBUG
      case L'i': buf_insert_char(buf, (unsigned char)buf->n); break;

      // size
      case L'p': g_dpi_scale += n; goto redraw_request;
      case L'P': g_dpi_scale -= n; if (g_dpi_scale < 4) g_dpi_scale = 4; goto redraw_request;
      case CTRL('p'): g_dpi_scale = (int)(GetDpiForWindow(g_bufs->h_wnd) / 14.4); goto redraw_request;
      case L't': g_font_height += n; goto redraw_request;
      case L'T': g_font_height -= n; if (g_font_height < 6) g_font_height = 6; goto redraw_request;
      case CTRL('t'): g_font_height = DEFAULT_FONT_HEIGHT;
redraw_request:
                      InvalidateRect(w, NULL, TRUE); break;

      // cursor
      case L'j': buf->cursor += n * 16; break;
      case L'k': buf->cursor -= n * 16; break;
      case L'l': buf->cursor += n; break;
      case L'h': buf->cursor -= n; break;
      case L'H': buf->cursor = (buf->pos + n - 1) * 16; break;
      case L'M': buf->cursor = (buf->pos + n_lines / 2) * 16; break;
      case L'L': buf->cursor = (buf->pos + n_lines - n) * 16; break;
      case L'w': buf->cursor = (buf->cursor / 4 + 1) * 4; break;
      case L'b': buf->cursor = (buf->cursor - 1) / 4 * 4; break;
      case L'^': buf->cursor &= ~0xf; break;
      case L'+': buf->cursor = (buf->cursor & ~0xf) + 16; break;
      case L'-': buf->cursor = (buf->cursor & ~0xf) - 16; break;

      // window / far jump
      case CTRL('j'): // FALLTHRU
      case L'e': buf->cursor += 16 * n; buf->pos += n; break;
      case CTRL('k'): // FALLTHRU
      case L'y': buf->cursor -= 16 * n; buf->pos -= n; break;
      case CTRL('f'): buf->cursor += n * n_lines * 16; buf->pos += n * n_lines; break;
      case CTRL('b'): buf->cursor -= n * n_lines * 16; buf->pos -= n * n_lines; break;
      case L'd': buf->cursor += n * n_lines / 2 * 16; buf->pos += n * n_lines / 2; break;
      case L'u': buf->cursor -= n * n_lines / 2 * 16; buf->pos -= n * n_lines / 2; break;
      case L'J': buf->pos += n * buf->memsize / 160; buf->cursor = buf->pos * 16; break;
      case L'K': buf->pos -= n * buf->memsize / 160; buf->cursor = buf->pos * 16; break;
      case CTRL('M'): buf->pos = buf->cursor / 16 + 1 - n; break;
      case L'g': // FALLTHRU
      case L'<': buf->pos = (buf->cursor = buf->n) / 16; break;
      case L'G': // FALLTHRU
      case L'>': buf->pos = (buf->cursor = buf->n ?: buf->memsize - 1) / 16 - n_lines + 1; break;

      }
      // reset
      buf->n = buf->m = 0;
      buf->input_choice = INPUT_CHOICE_NORMAL;

      // adjust
      if (buf->pos < 0) buf->pos = 0;
      if (buf->pos * 16 > buf->memsize) buf->pos = buf->memsize / 16;
      if (buf->cursor < 0) buf->cursor = 0;
      if (buf->cursor > buf->memsize) buf->cursor = buf->memsize;
      if (buf->cursor < buf->pos * 16) buf->pos = buf->cursor / 16;
      if (buf->cursor >= (buf->pos + n_lines) * 16) buf->pos = buf->cursor / 16;
      break;
   case WM_DESTROY:
      delete_buffer(buf);
      if (!g_n_bufs) PostQuitMessage(0);
      break;
   case WM_DROPFILES:
      open_files((HDROP)wp);
      break;
   case WM_LBUTTONDOWN:
      InvalidateRect(w, NULL, FALSE);
      if (wp & MK_SHIFT) buf->pos += n_lines;
      else if (wp & MK_CONTROL) buf->pos += buf->memsize / 160; else buf->pos++;
      if (buf->pos * 16 > buf->memsize) buf->pos = buf->memsize / 16;
      break;
   case WM_RBUTTONDOWN:
      InvalidateRect(w, NULL, FALSE);
      if (wp & MK_SHIFT) buf->pos -= n_lines;
      else if (wp & MK_CONTROL) buf->pos -= buf->memsize / 160; else buf->pos--;
      if (buf->pos < 0) buf->pos = 0;
      break;
   case WM_MOUSEWHEEL:
      InvalidateRect(w, NULL, FALSE);
      buf->pos -= (short)HIWORD(wp) / (WHEEL_DELTA / 3);
// fprintf(stderr, "(%d %d)\n", (short)HIWORD(wp), (short)HIWORD(wp) / (WHEEL_DELTA / 3));
      if (buf->pos < 0) buf->pos = 0;
      if (buf->pos * 16 > buf->memsize) buf->pos = buf->memsize / 16;
      break;
   case WM_PAINT: {
      PAINTSTRUCT ps;
      const HDC h_dc = BeginPaint(w, &ps);
      dump(h_dc, buf);
      EndPaint(w, &ps);
      break; }
   case WM_SIZE:
      buf->win_width  = LOWORD(lp);
      buf->win_height = HIWORD(lp);
      buf->win_height = HIWORD(lp) - 32 /*DEBUG*/;
      break;
   case WM_SETTINGCHANGE:
   case WM_THEMECHANGED: {
      InvalidateRect(w, NULL, FALSE);
      g_darkmode = is_darkmode();
      HBRUSH brush_bg = g_darkmode ? g_brush_dark_bg : (HBRUSH)GetStockObject(WHITE_BRUSH);
      SetClassLongPtr(w, GCLP_HBRBACKGROUND, (LONG_PTR)brush_bg); }
      break;
   default: return DefWindowProc(w, m, wp, lp);
   }
   return 0;
}

int
register_bmain_wc(HINSTANCE h_inst)
{
   g_brush_dark_bg = CreateSolidBrush(dark_bg);
   HBRUSH brush_bg = g_darkmode ? g_brush_dark_bg : (HBRUSH)GetStockObject(WHITE_BRUSH);

   WNDCLASS wc = {
      .hCursor = LoadCursor(NULL, IDC_ARROW),
      .hInstance = h_inst,
      .hbrBackground = brush_bg,
      .lpfnWndProc = wproc,
      .lpszClassName = B_MAIN,
      .style = CS_HREDRAW | CS_VREDRAW,
   };
   return RegisterClass(&wc);
}

PCTSTR
basename(PCTSTR s)
{
   for (PCTSTR i = s; *i; i++)
      if (*i == L'\\') s = i + 1;

   return s;
}

HWND
create_bmain_window(HINSTANCE h_inst, PCTSTR filename)
{
   const int x = 200, w = 1600 * g_dpi_scale / 10.0;
   const int y = 200, h = 800;
   const DWORD style = WS_OVERLAPPEDWINDOW;
   const HWND parent = NULL;
   const HMENU menu = NULL;
   const PVOID lp = NULL;

   TCHAR title[64];
   swprintf(title, 64, TEXT("Puting - %s"), basename(filename));

   HWND h_wnd = CreateWindow(B_MAIN,
         title, style, x, y, w, h, parent, menu, h_inst, lp);
   DragAcceptFiles(h_wnd, TRUE);
   return h_wnd;
}

int
mapping(struct buffer *buf)
{
   HANDLE f = CreateFile(buf->filename, GENERIC_READ, FILE_SHARE_READ, NULL,
         OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
   if (f == INVALID_HANDLE_VALUE) return 1;

   HANDLE fm = CreateFileMapping(f, NULL, PAGE_READONLY, 0, 0, NULL);
   if (!fm) return 1;

   buf->orig_mem = (char *)MapViewOfFile(fm, FILE_MAP_READ, 0, 0, 0);
   if (!buf->orig_mem) return 1;

#if 0
   MEMORY_BASIC_INFORMATION mbi;
   if (!VirtualQuery(buf->orig_mem, &mbi, sizeof mbi)) return 1;
   buf->memsize = mbi.RegionSize;
#endif

   LARGE_INTEGER filesize;
   if (GetFileSizeEx(f, &filesize))
      buf->memsize = filesize.QuadPart;

   buf->table = (struct piece *)malloc(sizeof(*buf->table));
   if (!buf->table) return 1;
   buf->table->which = PIECE_ORIG;
   buf->table->offs = 0;
   buf->table->size = buf->memsize;
   buf->tablesize = 1;

   CloseHandle(f);
   return 0;
}

int
is_darkmode(void)
{
   PCTSTR key = L"Software\\Microsoft\\Windows\\CurrentVersion\\Themes\\Personalize";
   PCTSTR name = L"AppsUseLightTheme";

   HKEY h_key;
   if (RegOpenKeyEx(HKEY_CURRENT_USER, key, 0, KEY_READ, &h_key) != ERROR_SUCCESS) {
      MessageBox(NULL, TEXT("RegOpenKeyEx Error"), TEXT("Puting"), MB_OK);
      return 0; }

   DWORD mode; /* 0=dark 1=light */
   DWORD len = 4;
   if (RegQueryValueEx(h_key, name, NULL, NULL, (PBYTE)&mode, &len) != ERROR_SUCCESS) {
      MessageBox(NULL, TEXT("RegQueryValueEx Error"), TEXT("Puting"), MB_OK);
      mode = 1; /* default is light */ }

   RegCloseKey(h_key);
   return mode == 0;
}

int
setup(int argc, wchar_t *argv[])
{
   // per process init
   g_darkmode = is_darkmode();
   if (!register_bmain_wc(g_hinst)) return 1;

   // per buffer init
   for (int i = 0; i < argc; i++) {
      struct buffer *buf = create_buffer();
      buf->filename = argv[i];
      if (mapping(buf)) {
         MessageBox(NULL, argv[i], TEXT("Puting: failed to open file"), MB_OK);
         delete_buffer(buf);
         continue; }

      HWND h_wnd = create_bmain_window(g_hinst, buf->filename);
      if (!h_wnd) {
         MessageBox(NULL, argv[i], TEXT("Puting: failed to open window"), MB_OK);
         delete_buffer(buf); }
      else {
         buf->h_wnd = h_wnd;
         ShowWindow(h_wnd, SW_SHOW); } }

   if (!g_n_bufs) {
      struct buffer *buf = create_buffer();
      buf->filename = L"(null)";
      HWND h_wnd = create_bmain_window(g_hinst, buf->filename);
      if (!h_wnd) {
         MessageBox(NULL, L"Puting: failed to open window", L"Puting", MB_OK);
         delete_buffer(buf); }
      else {
         buf->h_wnd = h_wnd;
         ShowWindow(h_wnd, SW_SHOW); } }

   if (g_n_bufs)
      g_dpi_scale = (int)(GetDpiForWindow(g_bufs->h_wnd) / 14.4);

   return 0;
}

int WINAPI
mainloop(void)
{
   MSG mesg;
   while (GetMessage(&mesg, NULL, 0, 0)) {
      TranslateMessage(&mesg);
      DispatchMessage(&mesg); }
   return mesg.wParam;
}

int WINAPI
wWinMain(
HINSTANCE h_inst,
HINSTANCE h_previnst,
PTSTR     s_cmdline,
int       e_cmdshow)
{
   int argc;
   LPTSTR *argv = CommandLineToArgvW(s_cmdline, &argc);
   if (*s_cmdline == '\0') argc = 0;

   SetThreadDpiAwarenessContext(DPI_AWARENESS_CONTEXT_SYSTEM_AWARE);

   g_hinst = h_inst;

   if (setup(argc, argv)) return 1;

   return mainloop();
}
// vim:sw=3:expandtab:
